/***************************************************************/
/*                                                             */
/* File name:   i2c_function	.c		                           */
/*                                                             */
/* Since:     2009-February-18                                 */
/*                                                             */
/*                                                             */
/* Author:    Designed by Rudy HOUQUE [RH]                     */
/*			  		Appli team RFID (EEPROM)								         */
/*            ISFEN    P.17	  (SCHOOL)                         */
/*                                                             */
/* Purpose:   This document use i2c library function to create */
/*						standar i2c frames for read/write temperature  	 */
/*            sensor and eeprom                                */
/*                                                             */
/* History:                                                    */
/* the 2009/02/18 [RH]  Created this file.                     */
/*                                                             */
/*                                                             */
/***************************************************************/

/* Includes ---------------------------------------------------*/
#include "i2c_function.h"
#include "stm8l10x_i2c.h"


/* Private typedef --------------------------------------------*/
/* Private define ---------------------------------------------*/
/* Private macro ----------------------------------------------*/
/* Private variables ------------------------------------------*/
	 uint8_t nb_of_read_byte =0;
	 uint8_t nb_of_transmitted_byte =0;
	 uint8_t i=0;	
   uint8_t j=0;
	 uint8_t error_code =0; 
/* Private function prototypes --------------------------------*/
/* Private functions ------------------------------------------*/


/***************************************************************/
/***********************  DELAY I2C  ******************************/
/***************************************************************/
void Delay_i2c(uint16_t nCount)
{
    while (nCount != 0)
    {
        nCount--;
    }
}
/***************************************************************/
/************  FUNCTION TO WRITE  I2C DATA  ***********************/
/***************************************************************/

/***************************************************************/
/**************  FUNCTION TO WRITE  I2C DATA  ******************/
/***************************************************************/
/*										    								  									 */
/*		Add_Slave : I2C Address of the slave component					 */
/*		Add_Eeprom : EEPROM adress to access					           */
/*		Data      : array of data to write in Add_Eeprom				 */
/*																			  										 */
/*      Return   : an error code see the correspondance        */
/*                 0 no error                                  */
/*                 1 the array Data or Data[0] is empty        */
/*                 2 only part of the array data was written   */
/*                                                             */
/*		Data[0]= 0x..	Number of transmit byte								     */
/*		Data[0]= 0x..	first byte of the data							       */
/*		Data[n]= 0x..	n Byte of the data							           */
/*                                                             */
/*    Example : if you Want to write the sequence 0x23 0x2A    */
/*     you have to send the array Data{0x02, 0x23, 0x2A,}      */
/*                                                             */
/***************************************************************/
int I2C_Write_Data (uint8_t Add_Slave, uint8_t *Add_Eeprom, uint8_t *Data)
{
	
	if(Data[0] !=0 ) /*if the data is not empty*/
	{
		/*------------------- START CONDITION ---------------------*/
		I2C_GenerateSTART(ENABLE);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_START_SENT));
		
		/*--------------- DEVICE SELECT + R/W BIT -----------------*/
		I2C_Send7bitAddress(Add_Slave, I2C_DIRECTION_TX);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_ADDRESS_ACKED));
    I2C_ClearFlag(I2C_FLAG_ADDRESSSENTMATCHED);
  
		/*---------------- SEND 2 BYTES EEPROM ADDRESS ------------*/
    I2C_SendData(Add_Eeprom[0]);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));
		I2C_SendData(Add_Eeprom[1]);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));
		
		/*------------------ SEND DATA TO WRITE -------------------*/
		nb_of_transmitted_byte=0;
    
		for(i = 0 ; i<=Data[0] ; i++)
		{
			I2C_SendData(Data[i+1]);
			while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));
			nb_of_transmitted_byte++;  
		}
		if(nb_of_transmitted_byte != Data[0]) 
		{
			error_code = 2;
		}
		
		/*-------------------- STOP CONDITION ---------------------*/
		I2C_GenerateSTOP(ENABLE);
		Delay_i2c(65000);//to let the eeprom the time to write
		error_code = 0;
	}/*if(Data[0] !=0 )*/
	else error_code = 1;
    
	return error_code;
}

/***************************************************************/
/**************  FUNCTION TO READ  I2C DATA  *******************/
/***************************************************************/

/***************************************************************/
/**************  FUNCTION TO READ  I2C DATA  **********************/
/***************************************************************/
/*										    								  									 */
/*		Add_Slave : I2C Address of the slave component					 */
/*		Add_Eeprom : EEPROM adress to access					           */
/*		Data      : array of data to read in Add_Eeprom 				 */
/*																			  										 */
/*      Return   : an error code see the correspondance        */
/*                 0 no error                                  */
/*                 1 the array Data or Data[0] is empty        */
/*                 2 only part of the array data was read      */
/*                                                             */
/*		Data[0]= 0x..	Number of bytes to be read						     */
/*		Data[0]= 0x..	first byte of the data							       */
/*		Data[n]= 0x..	n Byte of the data							           */
/*                                                             */
/*    Example : if you Want to read 5 bytes     							 */
/*     you have to send the array Data{0x05, 0x00, 0x00,...}   */
/*                                                             */
/***************************************************************/
int I2C_Read_Data (uint8_t Add_Slave, uint8_t *Add_Eeprom, uint8_t *Data)
{
	
	if(Data[0] != 0) /*if the data is not empty*/
	{
		/*----------------- START CONDITION -----------------------*/
		I2C_GenerateSTART(ENABLE);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_START_SENT));/*EV5*/
	
		/*------------- DEVICE SELECT + R/W BIT -------------------*/
		I2C_Send7bitAddress(Add_Slave, I2C_DIRECTION_TX);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_ADDRESS_ACKED));/*EV6*/
    I2C_ClearFlag(I2C_FLAG_ADDRESSSENTMATCHED);/*EV8_1*/
	
		/*---------------- SEND 2 BYTES EEPROM ADDRESS ------------*/
    I2C_SendData(Add_Eeprom[0]);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));
		I2C_SendData(Add_Eeprom[1]);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));

		/*----------------- START CONDITION -----------------------*/
		I2C_GenerateSTART(ENABLE);
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_START_SENT));/*EV5*/

		/*------------- DEVICE SELECT + R/W BIT -------------------*/
    I2C_Send7bitAddress(Add_Slave, I2C_DIRECTION_RX);
    while(!I2C_CheckEvent(I2C_EVENT_MASTER_ADDRESS_ACKED));/*EV6*/
    I2C_ClearFlag(I2C_FLAG_ADDRESSSENTMATCHED);
	
		/*----------- RECEIVE BYTE DATA + ACKNOLEDGE --------------*/
	
		I2C_AcknowledgeConfig(I2C_ACK_CURR);
	
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_RECEIVED));/*EV7*/
		Data[1] = (u8)(I2C_ReceiveData());
	
		I2C_GenerateSTOP(ENABLE);/*EV7_1*/
		I2C_AcknowledgeConfig(I2C_ACK_NONE);
	
		while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_RECEIVED));/*EV7*/
		Data[2] = (u8)(I2C_ReceiveData());	
	
		Delay_i2c(65000);
	
	}/*end of if(Data[0] !=0 )*/
	return 0;
}

/***************************************************************/
/**************  FUNCTION TO CONFIGURE SENSOR i2C **************/
/***************************************************************/
/***************************************************************/
/*										    				   													 */
/*		Add_Slave : I2C Address of the slave component		   		 */
/*		Pointer_Data : data to access and configure pointer	   	 */
/*															   														 */
/*      Return   : an error code see the correspondance        */
/*                 0 no error                                  */
/*                                                             */
/*		Pointer_Data[0]= 0x..	register pointer to access	   		 */
/*		Pointer_Data[1]= 0x..	register configuration		   			 */
/*                                                             */
/*    Example : if you Want to make a one-shot temperature     */
/*              mesure Pointer_Data[0] = 0x01                  */
/*              Pointer_Data[1] = 0x83                         */
/*                                                             */
/***************************************************************/
int I2C_Configure_Sensor (uint8_t Add_Slave, uint8_t *Pointer_Data)
{

	/*------------------- START CONDITION ---------------------*/
	I2C_GenerateSTART(ENABLE);
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_START_SENT));
	
	
	/*--------------- DEVICE SELECT + R/W BIT -----------------*/
	I2C_Send7bitAddress(Add_Slave, I2C_DIRECTION_TX);
	while(!I2C_CheckEvent(I2C_EVENT_SLAVE_ADDRESS_MATCHED));
	I2C_ClearFlag(I2C_FLAG_ADDRESSSENTMATCHED);
	
	/*---------------- SEND POINTER SELECTION BYTE ------------*/
	I2C_SendData(Pointer_Data[0]);
	while(!I2C_CheckEvent(I2C_EVENT_SLAVE_BYTE_TRANSMITTED ));
	/*---------------- SEND POINTER CONFIGURATION BYTE --------*/
	I2C_SendData(Pointer_Data[1]);
	while(!I2C_CheckEvent(I2C_EVENT_SLAVE_BYTE_TRANSMITTED ));
	
	/*-------------------- STOP CONDITION ---------------------*/
	I2C_GenerateSTOP(ENABLE);
	Delay_i2c(65000);//to let the eeprom the time to write
	error_code = 0;
	
	return error_code;
}



/***************************************************************/
/**************  FUNCTION TO READ  I2C DATA  *******************/
/***************************************************************/
/*										    				   													 */
/*		Add_Slave : I2C Address of the slave component		   		 */
/*		Pointer_Data : data to access  pointer	               	 */
/*		Temperature_Data : 2 Bytes temperature format		   			 */
/*																			  										 */
/*      Return   : an error code see the correspondance        */
/*                 0 no error                                  */
/*                 2 temperature data was not fully read       */
/*                                                             */
/*		Pointer_Data[0]= 0x..	register pointer to access	   		 */
/*		Pointer_Data[1]= 0x..	useless	                       		 */
/*                                                             */
/*      Temperature_data[0] : first part of 2 bytes            */
/*                            temperature format               */
/*      Temperature_data[1] : last part of 2 bytes             */
/*                            temperature format               */
/*                                                             */
/*    Example : if you Want to read the temperature stored     */
/*     you have to send following arrays                       */
/*              Pointer_Data[0] = 0x00                         */
/*              Pointer_Data[1] = 0x.. (useless)               */
/*                                                             */
/***************************************************************/
int I2C_Read_Temperature (uint8_t Add_Slave, uint8_t Pointer_Data, uint8_t *Temperature_Data)
{
	
	/*----------------- START CONDITION -----------------------*/
	I2C_GenerateSTART(ENABLE);
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_START_SENT));
	
	/*------------- DEVICE SELECT + R/W BIT -------------------*/
	I2C_Send7bitAddress(Add_Slave, I2C_DIRECTION_TX);
	/* ACK by STTS75 */
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_ADDRESS_ACKED));
	I2C_ClearFlag(I2C_FLAG_ADDRESSSENTMATCHED);
	
	/*---------------- SEND POINTER SELECTION BYTE ------------*/
	I2C_SendData(Pointer_Data);
	/* ACK by STTS75 */
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));
		
	/*----------------- START CONDITION -----------------------*/
	I2C_GenerateSTART(ENABLE);
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_START_SENT));

	/*------------- DEVICE SELECT + R/W BIT -------------------*/
	I2C_Send7bitAddress(Add_Slave, I2C_DIRECTION_RX);
	/* ACK by STTS75 */
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_ADDRESS_ACKED));
	I2C_ClearFlag(I2C_FLAG_ADDRESSSENTMATCHED);
	
	/*---- RECEIVE 2 BYTES TEMPERATURE DATA + ACKNOLEDGE ------*/
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_RECEIVED));
	Temperature_Data[0] = (u8)(I2C_ReceiveData());
	
	I2C_AcknowledgeConfig(I2C_ACK_NONE);
	
	while(!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_RECEIVED));
	Temperature_Data[1] = (u8)(I2C_ReceiveData());
	I2C_GenerateSTOP(ENABLE);	
	
	Delay_i2c(65000);
	
	return 0;
}

